<?php

namespace App\Http\Controllers\Backend;

use App\Account;
use App\Balance;
use App\Currency;
use App\Http\Controllers\Controller;
use App\Http\Controllers\Mail\RejectDWEmail;
use App\Http\Controllers\Mail\VerifyOrder;
use App\Order;
use App\Setting;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Kraken;
use TCV;

class BalanceController extends Controller
{

    //Execute if user is authenticated
    public function __construct()
    {

        $this->middleware(['auth', '2fa', 'verify.ip']);

    }

    //Get Client IP
    private function get_client_ip_server()
    {

        $ipaddress = '';

        if ($_SERVER['REMOTE_ADDR']) {

            $ipaddress = $_SERVER['REMOTE_ADDR'];

        } else {

            $ipaddress = 'UNKNOWN';

        }
        return $ipaddress;
    }

    //Function for order arrays
    private function sorting($order, $key)
    {

        return function ($a, $b) use ($order, $key) {

            if ($order == 'DESC') {
                if (empty($key)) {

                    return strnatcmp($a->amount, $b->amount);

                } else {

                    return strnatcmp($a->$key, $b->$key);

                }

            } else {

                if (empty($key)) {

                    return strnatcmp($b->amount, $a->amount);

                } else {

                    return strnatcmp($b->$key, $a->$key);

                }
            }
        };
    }

    //Function for Check if a url have a successfully response
    private function url_exists($url = null)
    {

        if (empty($url)) {
            return false;
        }

        $ch = curl_init($url);

        //Establecer un tiempo de espera
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);

        //establecer NOBODY en true para hacer una solicitud tipo HEAD
        curl_setopt($ch, CURLOPT_NOBODY, true);
        //Permitir seguir redireccionamientos
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        //recibir la respuesta como string, no output
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $data = curl_exec($ch);

        //Obtener el código de respuesta
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        //cerrar conexión
        curl_close($ch);

        //Aceptar solo respuesta 200 (Ok), 301 (redirección permanente) o 302 (redirección temporal)
        $accepted_response = array(200, 301, 302);

        if (in_array($httpcode, $accepted_response)) {

            return true;

        } else {

            return false;

        }
    }

    //Function For Generate Random Number
    private function generate($longitud)
    {
        $key = '';
        $pattern = '1234567890';
        $max = strlen($pattern) - 1;
        for ($i = 0; $i < $longitud; $i++) {
            $key .= $pattern{mt_rand(0, $max)};
        }

        return $key;
    }

    //Get Total Balance for clients in USD and BTC
    public function total()
    {

        //Select User
        $user = Auth::user();

        //Create $balances array
        $balances = array();

        //Verify If User Has Client Role
        if ($user->hasCredential('158')) {

            //Select Balances greater than 0
            $balancesP = Balance::Where('user_id', null)->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.*', 'symbol', 'value', 'currencies.type', 'name', 'currencies.price')->get();
        } else {
            $balancesP = Balance::Where('user_id', $user->id)->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.*', 'symbol', 'value', 'currencies.type', 'name', 'currencies.price')->get();
        }

        $count = 0;

        foreach ($balancesP as $balance) {

            if (empty($balances[$count])) {

                $balances[$count] = new \stdClass();

                $balances[$count]->amount = $balance->amount;
                $balances[$count]->value = $balance->value;
                $balances[$count]->symbol = $balance->symbol;
                $balances[$count]->type = $balance->type;
                $balances[$count]->name = $balance->name;
                $balances[$count]->price = $balance->price;
            } else {

                foreach ($balances as $bal) {

                    if ($bal->symbol == $balance->symbol) {

                        $newBals = $bal->amount + $balance->amount;
                        $bal->amount = $newBals;

                    }
                }

            }

            $count += 1;

        }

        //Declare $usd and $btc variables
        $usd = 0;
        $btc = 0;

        //Loop $balances array
        foreach ($balances as $balance) {

            $balance->value = $balance->price;
            $usdvalue = $balance->amount * $balance->value;

            //Assign USD and BTC values
            $usd += $usdvalue;
            if ($balance->symbol == 'BTC') {
                $btcprice = $balance->price;
            }

        }

        $btc = $usd / $btcprice;

        //Return Response in JSON Datatype
        return response()->json(['usd' => $usd, 'btc' => $btc], 202);
    }

    //Listing Balances
    public function index(Request $request)
    {

        //Select Authenticated User
        $user = Auth::User();

        //Declare Variables
        $searchValue = $request->searchvalue;
        $page = $request->page;
        $resultPage = $request->resultPage;
        $orderBy = $request->orderBy;
        $orderDirection = $request->orderDirection;
        $total = 0;

        //Declare $balancesCurrency array
        $balancesCurrency = array();

        //Declare $count variable
        $count = 0;

        //Verify If User has client role
        if ($user->hasCredential('157')) {
            //Select Balances
            $query = Balance::Where('balances.type', 'fund')->where('user_id', null)->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.*', 'symbol', 'value', 'currencies.type', 'name', 'currencies.exchangeable', 'currencies.price')->orderBy('amount', 'desc');
        } else {
            //Select Balances
            $query = Balance::Where('balances.type', 'fund')->where('user_id', $user->id)->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.*', 'symbol', 'value', 'currencies.type', 'name', 'currencies.exchangeable', 'currencies.price')->orderBy('amount', 'desc');
        }

        //Search by
        if ($searchValue != '') {
            $query->Where(function ($query) use ($searchValue) {
                $query->Where('symbol', 'like', '%' . $searchValue . '%')
                    ->orWhere('amount', 'like', '%' . $searchValue . '%')
                    ->orWhere('pending_amount', 'like', '%' . $searchValue . '%')
                    ->orWhere('value', 'like', '%' . $searchValue . '%');
            });
        }

        if ($resultPage == null || $resultPage == 0) {

            $resultPage = 10;

        }

        //Get Total
        $total = $query->get()->count();

        if ($page > 1) {

            $query->offset(($page - 1) * $resultPage);

        }

        $query->limit($resultPage);

        //Get Balances
        $balancesCurrencyP = $query->get();

        //Loop Balances
        foreach ($balancesCurrencyP as $balanceP) {

            if (empty($balancesCurrency[$count])) {

                $balancesCurrency[$count] = new \stdClass();

                $balancesCurrency[$count]->amount = $balanceP->amount;
                $balancesCurrency[$count]->value = $balanceP->value;
                $balancesCurrency[$count]->symbol = $balanceP->symbol;
                $balancesCurrency[$count]->pending_amount = $balanceP->pending_amount;
                $balancesCurrency[$count]->type = $balanceP->type;
                $balancesCurrency[$count]->name = $balanceP->name;
                $balancesCurrency[$count]->exchangeable = $balanceP->exchangeable;
                $balancesCurrency[$count]->equivalent = 0;
                $balancesCurrency[$count]->price = $balanceP->price;

            } else {

                foreach ($balancesCurrency as $value) {

                    if ($value->symbol == $balanceP->symbol) {

                        $newBal = $value->amount + $balanceP->amount;
                        $value->amount = $newBal;
                    }
                }
            }
            $count += 1;
        }

        //Loop $balancesCurrency Array
        foreach ($balancesCurrency as $balance) {
            //Verify $balance Symbol
            if ($balance->symbol == "USD") {

                $balance->value = 1;

            }
            if ($balance->symbol == "VES") {

                $balance->value = 0.01;

            }

        }

        //Loop $balancesCurrency array
        foreach ($balancesCurrency as $balance) {

            //Declare $balance Equivalent
            $balance->value = $balance->price;

            $balance->equivalent = ($balance->amount + $balance->pending_amount) * $balance->value;
        }

        //Verify User Role
        if ($user->hasRole('20') || $user->hasRole('901')) {
            $eaccess = true;
        } else {
            $eaccess = false;
        }

        //Sort $balancesCurrency Array
        usort($balancesCurrency, $this->sorting($orderDirection, $orderBy));

        //Return Response in JSON datatype
        return response()->json(['page' => $page, 'result' => $balancesCurrency, 'total' => $total, 'eaccess' => $eaccess], 202);
    }

    //Make Deposit
    public function deposit(Request $request)
    {

        $request->validate([
            'type' => 'required|max:50',
            'currency' => 'required|max:50',
            'address' => 'required|max:50',
            'reference' => 'required|max:250',
            'amount' => 'required',
        ]);

        $user = Auth::User();
        $type = $request->type;
        $currency = Currency::Where('symbol', $request->currency)->first();
        $address = $request->address;
        $reference = $request->reference;
        $amount = $request->amount;
        $ip = $this->get_client_ip_server();

        $verify = Order::where('reference', $reference)->where('status', '<>', 'cancelled')->get();

        if ($verify->count()) {
            return response()->json(['error' => 'TXID o referencia registrado previamente, en caso de problemas contacte atencion al cliente.'], 403);
        }

        $balance = Balance::where('currency_id', $currency->id)->where('user_id', $user->id)->first();
        $balanceG = Balance::where('currency_id', $currency->id)->where('user_id', null)->first();

        $order = new Order;

        $order->in_amount = $amount;
        $order->reference = $reference;
        $order->type = 'deposit';
        $order->status = "pending";
        $order->ip = $ip;
        $order->ubication = 'test';
        $order->inCurrencyOrder()->associate($currency->id);
        $order->user()->associate($user);
        if ($type == 'FIAT') {
            $account = Account::Where('address', $address)->first();
            $order->account()->associate($account);
        }
        $order->save();

        $balance->pending_amount += $amount;
        $balance->save();

        $balanceG->pending_amount += $amount;
        $balanceG->save();

        //Return Response in JSON datatype
        return response()->json(['message' => 'success'], 202);
    }

    //Make Withdraw
    public function withdraw(Request $request)
    {
        $request->validate([
            'type' => 'required|max:50',
            'currency' => 'required|max:50',
            'address' => 'required|max:50',
            'amount' => 'required',
            'fee' => 'required',
        ]);

        $user = Auth::User();
        $type = $request->type;
        $fee = $request->fee;
        $currency = Currency::Where('symbol', $request->currency)->first();

        if ($currency->type == 'FIAT') {
            $address = Account::Where('address', $request->address)->where('user_id', $user->id)->first();
        } else {
            $address = Account::Where('key_account', $request->address)->where('user_id', $user->id)->first();
        }

        $month = Carbon::now()->daysInMonth;

        $equivalent = $user->config->withdraw_max_limit / $month;

        $equivalent = ($equivalent - $user->config->daily_withdraw) / $currency->price;

        $reference = $this->generate(9);

        $amount = $request->amount - $fee;

        $amountC = $request->amount;

        $ip = $this->get_client_ip_server();

        $balance = Balance::where('currency_id', $currency->id)->where('user_id', $user->id)->first();
        $balanceG = Balance::where('currency_id', $currency->id)->where('user_id', null)->first();

        if ($balance->amount >= $amount && $amount !== 0) {

            if ($amount >= $equivalent) {
                return response()->json(['message' => 'success', 'respond' => 'Limite del día exedido'], 202);
            }
            $order = new Order;
            $order->out_amount = $amount;

            $order->type = 'withdraw';
            $order->status = "pending";
            $order->ip = $ip;
            $order->ubication = 'test';
            $order->fee = $fee;

            $order->outCurrencyOrder()->associate($currency->id);
            $order->user()->associate($user);
            $order->account()->associate($address);

            $balance->amount -= $amountC;
            $balance->pending_amount += $amountC;
            $balance->save();

            $balanceG->amount -= $amountC;
            $balanceG->pending_amount += $amountC;
            $balanceG->save();

            if ($currency->type == 'Cryptocurrency') {
                $fee = $currency->withdraw_fee * 0.50;
                
                if ($currency->symbol == 'PTR') {
                    $tcv = TCV::withdrawPetroFunds('PTR', $address->address, $amount);
                    $order->reference = $tcv['id'];
                } 

            } else {
                $order->reference = $reference;
            }

            $amountwithdrawal = $request->amount * $currency->price;

            $user->config->daily_withdraw += $amountwithdrawal;
            $user->config->monthly_withdraw += $amountwithdrawal;

            $user->push();

            $order->save();
        } else {
            return response()->json(['message' => 'success', 'respond' => 'Fondos Insuficientes!'], 202);
        }
        //Return Response in JSON datatype
        return response()->json(['message' => 'success', 'respond' => 'Retiro Hecho con exito!'], 202);
    }

    public function availableV(Request $request)
    {
        $user = Auth::User();
        //Declare Variables
        $symbol = $request->symbol;
        $symbol2 = $request->symbol2;
        $limit = 0;
        //Select Currency Balance
        $balance = Balance::Where('balances.type', 'fund')->where('user_id', $user->id)->where('currencies.symbol', $symbol)->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.amount', 'balances.pending_amount', 'currencies.symbol')->first();
        if (($symbol != 'VES' && $symbol2 != 'VES')) {
            if (($symbol == 'PTR' || $symbol2 == 'PTR')) {

                if ($symbol == 'PTR') {
                    $pair = strtolower($symbol) . strtolower($symbol2);
                    $tcv = TCV::getOrderBook($pair);
                    $limit = 0;
                    foreach ($tcv['bids'] as $valor){
                        $limit += $valor['remaining_volume'];
                    }
                } else {
                    $pair = strtolower($symbol2) . strtolower($symbol);
                    $tcv = TCV::getOrderBook($pair);
                    $limit = 0;
                    foreach ($tcv['asks'] as $valor){
                        $limit += $valor['remaining_volume'];
                    }
                }

            }
        }

        //Return Response in JSON dataType
        return response()->json(['result' => $balance, 'result2' => $limit], 202);
    }

    //Show Available Balance for Withdraw
    public function available(Request $request)
    {

        $user = Auth::User();
        //Declare Variables
        $symbol = $request->symbol;
        $tradeable = $request->tradeable;
        //Select Currency Balance
        $balance = Balance::Where('balances.type', 'fund')->where('user_id', $user->id)->whereIn('currencies.symbol', [$symbol, $tradeable])->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.amount', 'balances.pending_amount', 'currencies.symbol')->get();

        //Return Response in JSON dataType
        return response()->json(['result' => $balance], 202);
    }

    public function availableB(Request $request)
    {

        $user = Auth::User();
        //Declare Variables
        $symbol = $request->currency;
        //Select Currency Balance
        $balance = Balance::Where('balances.type', 'fund')->where('user_id', $user->id)->whereIn('currencies.symbol', [$symbol])->leftJoin('currencies', 'currencies.id', '=', 'balances.currency_id')->where('currencies.active', true)->select('balances.amount', 'balances.pending_amount', 'currencies.symbol')->get();

        //Return Response in JSON dataType
        return response()->json(['result' => $balance], 202);
    }

    //List of Transactions
    public function viewDeposit(Request $request)
    {

        //Select Authenticated user
        $user = Auth::User();

        //Assign Variables
        $searchValue = $request->searchvalue;
        $page = $request->page;
        $resultPage = $request->resultPage;
        $orderBy = $request->orderBy;
        $orderDirection = $request->orderDirection;
        $total = 0;

        //Verify if $orderBy is empty
        if (empty($orderBy)) {

            $orderBy = 'created_at';

        }

        //Create $transactions array
        $transactions = array();

        //Create $count variable
        $count = 0;

        //Verify if user has client role
        if ($user->hasCredential('450')) {

            //Select Orders
            $query = Order::where('orders.type', 'deposit')->leftJoin('profiles', 'profiles.user_id', '=', 'orders.user_id')->leftJoin('currencies', 'currencies.id', '=', 'orders.in_currency')->leftJoin('accounts', 'accounts.id', '=', 'orders.account_id')->where('currencies.active', true)->select('orders.*', 'symbol', 'entity', 'fname', 'flastname')->orderBy('created_at', 'desc');

        } else {

            //Select Orders
            $query = Order::where('user_id', $user->id)->where('orders.type', 'deposit')->leftJoin('currencies', 'currencies.id', '=', 'orders.in_currency')->where('currencies.active', true)->select('orders.*', 'symbol')->orderBy('created_at', 'desc');

        }

        //Search by
        if ($searchValue != '') {

            $query->Where(function ($query) use ($searchValue) {
                $query->Where('in_amount', 'like', '%' . $searchValue . '%')
                    ->orWhere('reference', 'like', '%' . $searchValue . '%')
                    ->orWhere('status', 'like', '%' . $searchValue . '%')
                    ->orWhere('orders.created_at', 'like', '%' . $searchValue . '%')
                    ->orWhere('currencies.symbol', 'like', '%' . $searchValue . '%');
            });

        }

        if ($resultPage == null || $resultPage == 0) {

            $resultPage = 10;

        }

        //Get Total
        $total = $query->get()->count();

        if ($page > 1) {

            $query->offset(($page - 1) * $resultPage);

        }

        $query->limit($resultPage);

        //Get Fund Orders
        $transactionsP = $query->get();

        //Loop Fund Orders
        foreach ($transactionsP as $transaction) {

            //Verify if $transactions array is empty
            if (empty($transactions[$count])) {

                //Create object in $transactions array
                $transactions[$count] = new \stdClass();
                if ($user->hasCredential('450')) {
                    $transactions[$count]->user = $transaction->fname . ' ' . $transaction->flastname;
                }
                $transactions[$count]->amount = $transaction->in_amount;
                $transactions[$count]->symbol = $transaction->symbol;
                $transactions[$count]->created_at = $transaction->created_at;
                $transactions[$count]->ip = $transaction->ip;
                $transactions[$count]->ubication = $transaction->ubication;
                $transactions[$count]->reference = $transaction->reference;
                $transactions[$count]->status = $transaction->status;
                $transactions[$count]->updated_at = $transaction->updated_at;
                $transactions[$count]->rate = $transaction->rate;
                $transactions[$count]->id = $transaction->id;
                $transactions[$count]->entity = $transaction->entity;
            }

            $count += 1;

        }

        //Sort $transactions array
        usort($transactions, $this->sorting($orderDirection, $orderBy));

        //Loop $transcations array
        foreach ($transactions as $transaction) {

            //Change date to formatted string
            $newcreated = $transaction->created_at->toFormattedDateString();
            $newupdated = $transaction->updated_at->toFormattedDateString();

            //Put New Dates
            $transaction->created_at = $newcreated;
            $transaction->updated_at = $newupdated;
        }

        //Verify if user is trader or admin

        if ($user->hasCredential('451')) {

            //Return true for administrative access
            $eaccess = true;

        } else {

            //Return false for administrative access
            $eaccess = false;

        }

        //Return Response in JSON datatype
        return response()->json(['page' => $page, 'result' => $transactions, 'total' => $total, 'eaccess' => $eaccess], 202);
    }

    //List of Transactions
    public function viewWithdraw(Request $request)
    {

        //Select Authenticated user
        $user = Auth::User();

        //Assign Variables
        $searchValue = $request->searchvalue;
        $page = $request->page;
        $resultPage = $request->resultPage;
        $orderBy = $request->orderBy;
        $orderDirection = $request->orderDirection;
        $total = 0;

        //Verify if $orderBy is empty
        if (empty($orderBy)) {

            $orderBy = 'created_at';

        }

        //Create $transactions array
        $transactions = array();

        //Create $count variable
        $count = 0;

        //Verify if user has client role
        if ($user->hasCredential('450')) {

            //Select Orders
            $query = Order::where('orders.type', 'withdraw')->leftJoin('organizations', 'organizations.user_id', '=', 'orders.user_id')->leftJoin('profiles', 'profiles.user_id', '=', 'orders.user_id')->leftJoin('accounts', 'accounts.id', '=', 'orders.account_id')->leftJoin('currencies', 'currencies.id', '=', 'orders.out_currency')->select('orders.*', 'symbol', 'entity', 'address', 'fname', 'flastname', 'id_number', 'user_type', 'merchant_record')->orderBy('created_at', 'desc');

        } else {

            //Select Orders
            $query = Order::where('user_id', $user->id)->where('orders.type', 'withdraw')->leftJoin('currencies', 'currencies.id', '=', 'orders.out_currency')->select('orders.*', 'symbol')->orderBy('created_at', 'desc');

        }

        //Search by
        if ($searchValue != '') {

            $query->Where(function ($query) use ($searchValue) {
                $query->Where('out_amount', 'like', '%' . $searchValue . '%')
                    ->orWhere('reference', 'like', '%' . $searchValue . '%')
                    ->orWhere('status', 'like', '%' . $searchValue . '%')
                    ->orWhere('orders.created_at', 'like', '%' . $searchValue . '%')
                    ->orWhere('currencies.symbol', 'like', '%' . $searchValue . '%');
            });

        }

        if ($resultPage == null || $resultPage == 0) {

            $resultPage = 10;

        }

        //Get Total
        $total = $query->get()->count();

        if ($page > 1) {

            $query->offset(($page - 1) * $resultPage);

        }

        $query->limit($resultPage);

        //Get Fund Orders
        $transactionsP = $query->get();

        //Loop Fund Orders
        foreach ($transactionsP as $transaction) {

            //Verify if $transactions array is empty
            if (empty($transactions[$count])) {

                //Create object in $transactions array
                $transactions[$count] = new \stdClass();
                if ($user->hasCredential('450')) {
                    if ($transaction->user_type == 1) {
                        $transactions[$count]->user = $transaction->fname . ' ' . $transaction->flastname . ' CI:' . $transaction->id_number;
                    } else {
                        $transactions[$count]->user = $transaction->fname . ' ' . $transaction->flastname . ' RIF:' . $transaction->merchant_record;
                    }
                }
                $transactions[$count]->amount = $transaction->out_amount;
                $transactions[$count]->symbol = $transaction->symbol;
                $transactions[$count]->created_at = $transaction->created_at;
                $transactions[$count]->ip = $transaction->ip;
                $transactions[$count]->ubication = $transaction->ubication;
                $transactions[$count]->reference = $transaction->reference;
                $transactions[$count]->status = $transaction->status;
                $transactions[$count]->updated_at = $transaction->updated_at;
                $transactions[$count]->rate = $transaction->fee;
                $transactions[$count]->id = $transaction->id;
                $transactions[$count]->entity = $transaction->entity;
                $transactions[$count]->address = $transaction->address;

            }

            $count += 1;

        }

        //Sort $transactions array
        usort($transactions, $this->sorting($orderDirection, $orderBy));

        //Loop $transcations array
        foreach ($transactions as $transaction) {

            //Change date to formatted string
            $newcreated = $transaction->created_at->toFormattedDateString();
            $newupdated = $transaction->updated_at->toFormattedDateString();

            //Put New Dates
            $transaction->created_at = $newcreated;
            $transaction->updated_at = $newupdated;
        }

        //Verify if user is trader or admin

        if ($user->hasCredential('451')) {

            //Return true for administrative access
            $eaccess = true;

        } else {

            //Return false for administrative access
            $eaccess = false;

        }

        //Return Response in JSON datatype
        return response()->json(['page' => $page, 'result' => $transactions, 'total' => $total, 'eaccess' => $eaccess], 202);
    }

    public function exchange(Request $request)
    {

        $request->validate([
            'type' => 'required',
            'amountI' => 'required',
            'amountO' => 'required',
            'symbolI' => 'required|min:01',
            'symbolO' => 'required',
        ]);

        $user = Auth::user();
        $symbolI = $request->symbolI;
        $symbolO = $request->symbolO;
        $amountI = $request->amountI;
        $amountO = $request->amountO;

        $priceI = $amountI * $amountO;

        $inner = Currency::Where('symbol', $request->symbolI)->first();
        $outer = Currency::Where('symbol', $request->symbolO)->first();

        $reference = $this->generate(9);

        $order = new Order;

        if ($request->type == 'bid') {
            $balance = Balance::where('currency_id', $inner->id)->where('user_id', $user->id)->first();
            $balanceG = Balance::where('currency_id', $inner->id)->where('user_id', null)->first();

            if ($balance->amount > $priceI) {
                $order->out_amount = $priceI;
                $balance->amount -= $priceI;
                $balance->pending_amount += $priceI;
                $balanceG->amount -= $priceI;
                $balanceG->pending_amount += $priceI;
            } else {

                return response()->json(['error' => 'You don\'t have enough funds'], 403);

            }

            $order->fee = 0;
            $order->rate = $amountI;
            $order->in_amount = $amountO;
            $order->type = 'exchange';
            $order->trade_type = $request->type;
            $order->ip = $this->get_client_ip_server();
            $order->ubication = 'test';
            $order->reference = $reference;
            $order->status = 'pending';
            $order->outCurrencyOrder()->associate($inner);
            $order->inCurrencyOrder()->associate($outer);
            $order->user()->associate($user);

        } else if ($request->type == 'ask') {
            $balance = Balance::where('currency_id', $outer->id)->where('user_id', $user->id)->first();
            $balanceG = Balance::where('currency_id', $outer->id)->where('user_id', null)->first();

            if ($balance->amount > $amountO) {
                $order->out_amount = $amountO;
                $balance->amount -= $amountO;
                $balance->pending_amount += $amountO;
                $balanceG->amount -= $amountO;
                $balanceG->pending_amount += $amountO;
            } else {
                return response()->json(['error' => 'You don\'t have enough funds'], 403);
            }
            $order->fee = 0;
            $order->rate = $amountI;
            $order->in_amount = $priceI;
            $order->type = 'exchange';
            $order->trade_type = $request->type;
            $order->ip = $this->get_client_ip_server();
            $order->ubication = 'test';
            $order->status = 'pending';
            $order->reference = $reference;
            $order->outCurrencyOrder()->associate($outer);
            $order->inCurrencyOrder()->associate($inner);
            $order->user()->associate($user);
        }
        $order->save();
        $balance->save();
        $balanceG->save();

        $orderF = Order::find($order->id);
        if ($request->type == 'bid') {
            $orders = Order::Where('trade_type', 'ask')->whereNotIn('user_id', [$user->id])->where('status', 'pending')->orderBy('created_at', 'desc')->get();
        } else if ($request->type == 'ask') {
            $orders = Order::Where('trade_type', 'bid')->whereNotIn('user_id', [$user->id])->where('status', 'pending')->orderBy('created_at', 'desc')->get();
        }

        foreach ($orders as $ord) {
            if ($ord->rate == $amountI) {
                if ($ord->trade_type == 'bid') {
                    $orderFi = $orderF->in_amount;
                    $orderSe = $orderF->out_amount;
                    $parse = $ord->out_amount;
                    $parse2 = $ord->in_amount;
                } else {
                    $orderFi = $orderF->out_amount;
                    $orderSe = $orderF->in_amount;
                    $parse = $ord->in_amount;
                    $parse2 = $ord->out_amount;
                }
                if ($parse != $ord->filled) {
                    $fill = $ord->filled + $orderFi;
                    if ($parse < $fill) {
                        $difex = $fill - $parse;
                        $newPrifill = $parse - $difex;
                        $newFill = $ord->filled + $newPrifill;
                        $ord->filled = $newFill;
                        $orderF->filled += $newPrifill;

                        if (bccomp($ord->filled, $parse, 8) == 0) {
                            $ord->status = 'complete';

                            $balanceord = Balance::where('user_id', $ord->user_id)->where('currency_id', $ord->in_currency)->first();
                            $balanceGord = Balance::where('user_id', null)->where('currency_id', $ord->in_currency)->first();
                            $balanceordo = Balance::where('user_id', $ord->user_id)->where('currency_id', $ord->out_currency)->first();
                            $balanceGordo = Balance::where('user_id', null)->where('currency_id', $ord->out_currency)->first();

                            if ($ord->trade_type == 'bid') {
                                $balanceGordo->pending_amount -= $newPrifill;
                                $balanceordo->pending_amount -= $newPrifill;
                            } else {
                                $balanceGordo->pending_amount -= ($newPrifill / $ord->rate);
                                $balanceordo->pending_amount -= ($newPrifill / $ord->rate);
                            }
                            $balanceord->amount += $ord->in_amount;

                            $balanceGord->amount += $ord->in_amount;

                            if (!(bccomp($orderF->filled, $orderFi, 8) == 0)) {
                                $balanceF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $ord->in_currency)->first();
                                if ($ord->trade_type == 'bid') {
                                    $balanceF->pending_amount -= ($newPrifill / $ord->rate);
                                    $balanceGord->pending_amount -= ($newPrifill / $ord->rate);
                                } else {
                                    $balanceGord->pending_amount -= $newPrifill;
                                    $balanceF->pending_amount -= $newPrifill;
                                }
                                $balanceF->save();
                            }

                            $balanceord->Save();
                            $balanceordo->Save();
                            $balanceGord->Save();
                            $balanceGordo->Save();

                        }
                        $ord->save();

                        if (bccomp($orderF->filled, $orderF, 8) == 0) {
                            $orderF->status = 'complete';

                            $balanceordF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $orderF->in_currency)->first();
                            $balanceGordF = Balance::where('user_id', null)->where('currency_id', $orderF->in_currency)->first();
                            $balanceordoF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $orderF->out_currency)->first();
                            $balanceGordoF = Balance::where('user_id', null)->where('currency_id', $orderF->out_currency)->first();

                            if (!(bccomp($ord->filled, $parse, 8) == 0)) {
                                $balanceS = Balance::where('user_id', $ord->user_id)->where('currency_id', $orderF->in_currency)->first();
                                if ($orderF->trade_type == 'bid') {
                                    $balanceS->pending_amount -= ($newPrifill / $ord->rate);
                                    $balanceGordF->pending_amount -= ($newPrifill / $ord->rate);
                                } else {
                                    $balanceGordF->pending_amount -= $newPrifill;
                                    $balanceS->pending_amount -= $newPrifill;
                                }
                                $balanceS->save();
                            }

                            if ($orderF->trade_type == 'bid') {
                                $balanceGordo->pending_amount -= $orderF->filled;
                                $balanceordo->pending_amount -= $orderF->filled;
                            } else {
                                $balanceGordo->pending_amount -= ($orderF->filled / $ord->rate);
                                $balanceordo->pending_amount -= ($orderF->filled / $ord->rate);
                            }
                            $balanceordF->amount += $orderF->in_amount;

                            $balanceGordF->amount += $orderF->in_amount;

                            $balanceordF->Save();
                            $balanceordoF->Save();
                            $balanceGordF->Save();
                            $balanceGordoF->Save();
                            $orderF->save();
                            break;
                        }
                        $orderF->save();
                    } else if ($parse >= $fill) {

                        $ord->filled = $fill;
                        $orderF->filled = $orderFi;

                        if (bccomp($ord->filled, $parse, 8) == 0) {
                            $ord->status = 'complete';

                            $balanceord = Balance::where('user_id', $ord->user_id)->where('currency_id', $ord->in_currency)->first();
                            $balanceGord = Balance::where('user_id', null)->where('currency_id', $ord->in_currency)->first();
                            $balanceordo = Balance::where('user_id', $ord->user_id)->where('currency_id', $ord->out_currency)->first();
                            $balanceGordo = Balance::where('user_id', null)->where('currency_id', $ord->out_currency)->first();

                            if ((bccomp($orderF->filled, $orderFi, 8) == 0)) {
                                $balanceF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $ord->in_currency)->first();
                                if ($ord->trade_type == 'bid') {
                                    $balanceF->pending_amount -= ($orderF->filled / $ord->rate);
                                    $balanceGord->pending_amount -= ($orderF->filled / $ord->rate);
                                } else {
                                    $balanceGord->pending_amount -= $orderF->filled;
                                    $balanceF->pending_amount -= $orderF->filled;
                                }
                                $balanceF->save();
                            }

                            if ($ord->trade_type == 'bid') {
                                $balanceGordo->pending_amount -= $orderF->filled;
                                $balanceordo->pending_amount -= $orderF->filled;
                            } else {
                                $balanceGordo->pending_amount -= ($orderF->filled / $ord->rate);
                                $balanceordo->pending_amount -= ($orderF->filled / $ord->rate);
                            }
                            $balanceord->amount += $ord->in_amount;

                            $balanceGord->amount += $ord->in_amount;

                            $balanceord->Save();
                            $balanceordo->Save();
                            $balanceGord->Save();
                            $balanceGordo->Save();

                        }
                        $ord->save();

                        if (bccomp($orderF->filled, $orderFi, 8) == 0) {
                            $orderF->status = 'complete';

                            $balanceordF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $orderF->in_currency)->first();
                            $balanceGordF = Balance::where('user_id', null)->where('currency_id', $orderF->in_currency)->first();
                            $balanceordoF = Balance::where('user_id', $orderF->user_id)->where('currency_id', $orderF->out_currency)->first();
                            $balanceGordoF = Balance::where('user_id', null)->where('currency_id', $orderF->out_currency)->first();

                            if (!(bccomp($ord->filled, $parse, 8) == 0)) {
                                $balanceS = Balance::where('user_id', $ord->user_id)->where('currency_id', $orderF->in_currency)->first();
                                if ($orderF->trade_type == 'bid') {
                                    $balanceS->pending_amount -= ($orderF->filled / $ord->rate);
                                    $balanceGordF->pending_amount -= ($orderF->filled / $ord->rate);
                                } else {
                                    $balanceGordF->pending_amount -= $orderF->filled;
                                    $balanceS->pending_amount -= $orderF->filled;
                                }
                                $balanceS->save();
                            }

                            if ($orderF->trade_type == 'bid') {
                                $balanceGordoF->pending_amount -= $orderF->filled;
                                $balanceordoF->pending_amount -= $orderF->filled;
                            } else {
                                $balanceGordoF->pending_amount -= ($orderF->filled / $ord->rate);
                                $balanceordoF->pending_amount -= ($orderF->filled / $ord->rate);
                            }

                            $balanceordF->amount += $orderF->in_amount;

                            $balanceGordF->amount += $orderF->in_amount;

                            $balanceordF->Save();
                            $balanceordoF->Save();
                            $balanceGordF->Save();
                            $balanceGordoF->Save();
                            $orderF->save();
                            break;
                        }
                        $orderF->save();
                    }
                }
            }
        }

        return response()->json(['message' => 'Your order #' . $reference . ' was place with success'], 202);
    }

    public function fastexchange(Request $request)
    {

        $request->validate([
            'type' => 'required',
            'amountI' => 'required',
            'amountO' => 'required',
            'symbolI' => 'required|min:01',
            'symbolO' => 'required',
        ]);

        $user = Auth::user();
        $symbolI = $request->symbolI;
        $symbolO = $request->symbolO;
        $amountI = $request->amountI;
        $amountO = $request->amountO;

        if ($request->type == 1) {
            $type = 'bid';
        } else {
            $type = 'ask';
        }

        $inner = Currency::Where('symbol', $symbolI)->first();
        $outer = Currency::Where('symbol', $symbolO)->first();

        $reference = $this->generate(9);

        $order = new Order;

        $firstP = (strtolower($inner->symbol) == 'ves' || strtolower($inner->symbol) == 'ptr');
        $secondP = (strtolower($outer->symbol) == 'ves' || strtolower($outer->symbol) == 'ptr');
        $setting = Setting::First();

        $usdPrice = $amountI * $inner->price;

        if ($inner->type == 'Cryptocurrency' && $outer->type == 'FIAT') {
            $fee = $setting->fee_crypto_fiat;
        } else if ($inner->type == 'Cryptocurrency' && $outer->type == 'Cryptocurrency') {
            $fee = $setting->fee_crypto_crypto;
        } else {
            $fee = $setting->fee_fiat_crypto;
        }

        //if ($type == 'bid') {

        $balance = Balance::where('currency_id', $inner->id)->where('user_id', $user->id)->first();
        $balanceG = Balance::where('currency_id', $inner->id)->where('user_id', null)->first();

        if ($usdPrice < 10) {
            return response()->json(['error' => 'El monto minimo de transaccion son 10$ americanos'], 403);
        }
        
        $pair="";
        if(strtolower($inner->symbol)=='ptr')
            $pair = strtolower($inner->symbol) . strtolower($outer->symbol);
        else if(strtolower($outer->symbol)=='ptr')
            $pair = strtolower($outer->symbol) . strtolower($inner->symbol);
            
        if($pair!==""){   
            $kraken = TCV::getOrderBook($pair);
            $sum = 0;
            if($kraken!=""){
                if(strtolower($inner->symbol)=='ptr') {
                    foreach ($kraken['asks'] as $valor){
                        $sum += $valor['remaining_volume'];
                    }
                    if($sum < $amountI){
                        return response()->json(['error' => 'No existe suficiente volumen. Disponible '.$sum], 403);
                    }
                }else if(strtolower($outer->symbol)=='ptr') {
                    foreach ($kraken['bids'] as $valor){
                        $sum += $valor['remaining_volume'];
                    }
                    if($sum < $amountO){
                        return response()->json(['error' => 'No existe suficiente volumen. Disponible '.$sum], 403);
                    }
                }
            }
                    
            
        }
        
        if ($balance->amount > $amountI) {

            $balance->amount -= $amountI;
            $balance->pending_amount += $amountI;
            $balanceG->amount -= $amountI;
            $balanceG->pending_amount += $amountI;
        } else {

            return response()->json(['error' => 'Fondos insuficientes'], 403);

        }
        
        $feeO = $amountI * $fee;
        $order->fee = $feeO;
        $order->fee_exchange = $feeO;
        $order->fee_network = 0;
        $order->fee_thirdparty = 0;
        $order->out_amount = $amountI;
        $order->rate = 0;
        $order->in_amount = $amountO;
        $order->type = 'market';
        $order->trade_type = $type;
        $order->ip = $this->get_client_ip_server();
        $order->ubication = 'test';
        $order->reference = $reference;
        $order->status = 'pending';
        $order->outCurrencyOrder()->associate($inner);
        $order->inCurrencyOrder()->associate($outer);
        $order->user()->associate($user);

        /*  } else if ($type == 'ask') {

        $balance = Balance::where('currency_id', $outer->id)->where('user_id', $user->id)->first();
        $balanceG = Balance::where('currency_id', $outer->id)->where('user_id', null)->first();

        if ($balance->amount > $amountO) {
        $balance->amount -= $amountO;
        $balance->pending_amount += $amountO;
        $balanceG->amount -= $amountO;
        $balanceG->pending_amount += $amountO;
        } else {
        return response()->json(['error' => 'You don\'t have enough funds'], 403);
        }

        $feeO = $amountO * $fee;
        $order->fee = $feeO;
        $order->fee_exchange = $feeO;
        $order->out_amount = $amountO;
        $order->rate = 0;
        $order->in_amount = $amountI;
        $order->type = 'market';
        $order->trade_type = $type;
        $order->ip = $this->get_client_ip_server();
        $order->ubication = 'test';
        $order->status = 'pending';
        $order->reference = $reference;
        $order->outCurrencyOrder()->associate($outer);
        $order->inCurrencyOrder()->associate($inner);
        $order->user()->associate($user);
        }*/
        $order->save();
        $balance->save();
        $balanceG->save();
        $data = [
            'reference' => $reference,
            'in_cu' => $symbolI,
            'out_cu' => $symbolO,
            'in_amount' => $amountI,
            'out_amount' => $amountO,
            'name' => $user->profile->fname,
            'create' => $order->created_at,
        ];

        Mail::to(["sistemas@afx.trade"])->send(new VerifyOrder($data));

        return response()->json(['message' => 'Your order #' . $reference . ' was place with success'], 202);
    }

    public function receiptPDF(Request $request)
    {

    }

    //Validate Balance
    public function verify(Request $request)
    {

        //Validate $request data
        $request->validate([
            'id' => 'required',
        ]);

        //Declare Variables
        $id = $request->id;
        //Find Pending Transaction
        $order = Order::find($id);

        if ($order->type == 'deposit') {
            $balance = Balance::Where('user_id', $order->user_id)->where('currency_id', $order->in_currency)->first();
            $balanceG = Balance::Where('user_id', null)->where('currency_id', $order->in_currency)->first();

            $balance->pending_amount -= $order->in_amount;
            $balance->amount += $order->in_amount;

            $balanceG->pending_amount -= $order->in_amount;
            $balanceG->amount += $order->in_amount;
        } else {
            $balance = Balance::Where('user_id', $order->user_id)->where('currency_id', $order->out_currency)->first();
            $balanceG = Balance::Where('user_id', null)->where('currency_id', $order->out_currency)->first();

            $balanceG->pending_amount -= $order->out_amount + $order->fee;
            $balance->pending_amount -= $order->out_amount + $order->fee;

        }

        $order->status = 'complete';

        $order->save();
        $balance->save();
        $balanceG->save();

        //Return Response in JSON dataType
        return response()->json(['message' => 'Your Validation was processed successfully'], 202);
    }

    //Destroy Order
    public function destroyOrder(Request $request)
    {

        //Validate $request data
        $request->validate([

            'id' => 'required',

        ]);

        //Assign Variables
        $id = $request->id;
        //Find Fund Order
        $order = Order::find($id);

        $user = User::find($order->user_id);
        $account = Account::find($order->account_id);
        $bank = ($account) ? $account->entity : 'Sin Banco';

        if ($order->type == 'withdraw') {

            $currency = Currency::find($order->out_currency);
            $amount = $order->out_amount;
            $fee = $order->fee;
            $balanceU = Balance::Where('currency_id', $order->out_currency)->where('user_id', $order->user_id)->first();
            $balanceG = Balance::where('currency_id', $order->out_currency)->where('user_id', null)->first();

            if ($order->status == 'pending') {
                $balanceU->amount += $amount + $fee;
                $balanceU->pending_amount -= $amount + $fee;
                $balanceU->save();

                $balanceG->amount += $amount + $fee;
                $balanceG->pending_amount -= $amount + $fee;
                $balanceG->save();
            } else if ($order->status == 'complete') {
                $balanceU->amount += $amount + $fee;
                $balanceU->save();

                $balanceG->amount += $amount + $fee;
                $balanceG->save();
            }

            $data = ['type' => 'Retiro', 'reference' => $order->reference, 'amount' => $order->out_amount, 'bank' => $bank, 'currency' => $currency->name];

        } else if ($order->type == 'deposit') {
            $currency = Currency::find($order->in_currency);

            $amount = $order->in_amount;

            $balanceU = Balance::Where('currency_id', $order->in_currency)->where('user_id', $order->user_id)->first();

            $balanceG = Balance::where('currency_id', $order->in_currency)->where('user_id', null)->first();
            if ($order->status == 'pending') {
                $balanceU->pending_amount -= $amount;
                $balanceU->save();

                $balanceG->pending_amount -= $amount;
                $balanceG->save();
            } else if ($order->status == 'complete') {
                $balanceU->amount -= $amount;
                $balanceU->save();

                $balanceG->amount -= $amount;
                $balanceG->save();
            }
            $data = ['type' => 'Deposito', 'reference' => $order->reference, 'amount' => $order->in_amount, 'bank' => $bank, 'currency' => $currency->name];

        }

        Mail::to($user->email)->send(new RejectDWEmail($data));

        //Delete Order
        $order->status = "cancelled";
        $order->save();

        //Return response in Json Datatype
        return response()->json(['message' => 'Complete'], 202);
    }

    //Withdraw Information
    public function withdrawInfo(Request $request)
    {

        $request->validate([
            'currency' => 'required',
            'address' => 'required',
            'amount' => 'required',
        ]);

        $user = Auth::user();

        $currency = $request->currency;
        $address = $request->address;
        $amount = $request->amount;

        if ($amount == '') {
            $amount = 0;
        }

        $coin = Currency::where('symbol', $currency)->first();
        
        $percent = $coin->withdraw_fee;

        if (strtolower($coin->type) == 'cryptocurrency') {
            $fees = $coin->withdraw_fee + $percent;
            if ($currency == 'PTR') {

                $tcv = TCV::getFee('withdraw');

                foreach ($tcv as $fee) {
                    if ($fee['currency'] == 'ptr') {
                        $result = array();

                        $result['fee'] = $fees;
                        $result['limit'] = $amount;
                        $result['amount'] = $amount - $fees;
                    }
                }

            } /* KRAKEN else {
                if ($currency == 'BTC') {
                    $currency = 'XBT';
                }

                $kraken = Kraken::withdrawInfo($currency, $address, $amount);

                $kraken['result']['fee'] = $fees;
                $kraken['result']['amount'] = $amount - $fees;
                $result = $kraken['result'];
            }*/

        } else {

            $result = array();

            $result['fee'] = $coin->withdraw_fee;
            $result['limit'] = $amount;
            $result['amount'] = $amount - $coin->withdraw_fee;
        }

        $month = Carbon::now()->daysInMonth;
        $mLimit = $user->config->withdraw_max_limit;
        $limit = $mLimit / $month;
        $spent = $user->config->daily_withdraw;
        $mspent = $user->config->monthly_withdraw;

        $result['minimal'] = $coin->min_withdraw + $percent;
        $result['mlimit'] = $mLimit;
        $result['dlimit'] = $limit;

        if ($coin->symbol == "VES") {

            $result['limit'] = (($limit - $spent) / $coin->price > 0) ? ($limit - $spent) / $coin->price : 0;

        } else if ($coin->symbol == "PTR") {

            $result['limit'] = (($limit - $spent) / $coin->price > 0) ? ($limit - $spent) / $coin->price : 0;

        } else {
            $result['limit'] = (($limit - $spent) / $coin->price > 0) ? ($limit - $spent) / $coin->price : 0;
        }

        $result['spent'] = $spent;
        $result['mspent'] = $mspent;

        return response()->json(['result' => $result, 'message' => 'Complete'], 202);
    }

    public function completeOrder(Request $request)
    {

        $id = $request->id;
        $value = $request->value;
        $price = $request->price;

        $order = Order::find($id);

        $balanceO = Balance::Where('user_id', $order->user_id)->where('currency_id', $order->out_currency)->first();
        $balanceGO = Balance::Where('user_id', null)->where('currency_id', $order->out_currency)->first();

        $balanceI = Balance::Where('user_id', $order->user_id)->where('currency_id', $order->in_currency)->first();
        $balanceGI = Balance::Where('user_id', null)->where('currency_id', $order->in_currency)->first();

        $balanceO->pending_amount -= $order->out_amount;
        $balanceGO->pending_amount -= $order->out_amount;

        $balanceI->amount += $value;
        $balanceGI->amount += $value;

        $balanceO->save();
        $balanceGO->save();
        $balanceI->save();
        $balanceGI->save();

        if ($order->trade_type == 'bid') {

            $rate = $order->out_amount / $value;
            $price = 1 / $price;

        } else {

            $rate = $order->out_amount / $value;
            $price = $price;

        }

        $fee = $price - $rate;

        if ($fee < 0) {
            $fee = $rate - $price;
        }

        $order->status = 'complete';
        $order->filled = $order->out_amount;
        $order->rate = $price;
        $order->fee_network = 0;
        $order->fee += $fee;
        $order->fee_thirdparty = $fee;
        $order->in_amount = $value;
        $order->save();

        return response()->json(['result' => $rate, 'message' => 'Complete'], 202);
    }

}
